/*
 * This is a test program to demonstrate Real-Time DAQ in
 * Linux, using RTAI and Comedi.
 *
 * Based on RTAI 24.1.9 and Comedi 0.7.65
 * May work with other versions, though not with Comedi 0.7.64
 * or earlier.
 *
 * This example was made by Maarten Egmond
 */

/* This is a kernel module, and we need a few header files for
 * that
 */
#include <linux/kernel.h>
#include <linux/module.h>

/* Include relevant header files for rtai: the main rtai.h that
 * we always need, rtai_sched because we use rt_task_xxx
 * functions and additionally rtai_fifos.h because we also
 * use fifos.
 */
#include <rtai.h>
#include <rtai_sched.h>

/* Include relevant header files for comedi: kernel stuff */
#include <linux/comedilib.h>

/* Finally, include our own include file */
#include "realtimewave.h"

MODULE_LICENSE("GPL");
EXPORT_NO_SYMBOLS; /* Not entirely sure what this is - copied from other example code */

#define PERIOD 50000LL /* 50 microsecond period - don't overload the system too quickly */

/* All variables should be declared static */
static RT_TASK task_waveout;
static volatile int stop=0;
comedi_t *daq;

/* Variables for performance checking */
static volatile int errorcount=0;
static volatile int latecount=0;
static volatile int maxdelayns=0;
static volatile unsigned int loopcount=0;
static volatile int mindelayns=1000000000; /* start minimum at 1 sec */

/* This is a module, so we need an init_module() routine */
int init_module(void)
{
  RTIME tick_period;

  /* Print a message using printk so we can use this for debugging
   * To see this information, run 'dmesg' on a commandline
   */
  rt_printk("REALTIMEWAVE: Init_module() called\n");

  /* Get access to our daq card */
  daq=comedi_open("/dev/comedi0"); /* Device number 0: /dev/comedi0 */
  if (!daq)
  {
    rt_printk("Error - Can not open /dev/comedi0!\n");
    return 1;
  }

  /* Activate rtai - even though the module is loaded it is still 'sleeping' */
  rt_mount_rtai();

  /* Initialize our task - see documentation for details */
  if (rt_task_init(&task_waveout,task_waveout_run,0,2048,0,0,NULL))
  {
    rt_printk("Error - Can not initialize task!\n");
    comedi_close(daq);
    return 1;
  }

  /* Set up periodic mode (this is already default, but do it anyway) */
  rt_set_periodic_mode();

  /* Start the timer */
  tick_period=start_rt_timer((int)nano2count(PERIOD));

  /* Set up the task for periodic running (i.e. not just once) */
  if (rt_task_make_periodic(&task_waveout,rt_get_time()+10*tick_period,tick_period))
  {
    rt_printk("Error - Can not make task periodic\n");
    comedi_close(daq);
    return 1;
  }

  rt_printk("REALTIMEWAVE: Init_module() finished\n");
  return 0;
}

void cleanup_module(void)
{
  unsigned int h,m,s;

  rt_printk("REALTIMEWAVE: Cleanup_module() called\n");

  /* Stop the task nicely. It checks for 'stop' so set it here
   * to make the task realize it should quit at the next loop.
   * Then we wait a short while to make sure it's gone and then
   * stop the timer and kill the task (if still around). The amount
   * of 'busy_sleep' is chosen arbitrarily (following some of the
   * other examples). 100 microseconds should be enough since the
   * loop runs at 50 microsecond intervals, however if this is used
   * (or anywhere near it) the rt_printk() at the end of the loop
   * is not executed, so there is something else odd going on?
   */
  stop=1;
  rt_busy_sleep(100000000);
  stop_rt_timer();
  rt_busy_sleep(100000000);
  rt_task_delete(&task_waveout);

  /* Let RTAI sleep again */
  rt_umount_rtai();

  /* Close our DAQ device */
  comedi_close(daq);

  /* Calculate (approximate) time spent. Since loopcount counts
   * 1000's of loops, and we run at 50 microsec intervals, 20
   * loopcounts are in a single second.
   */
  h=loopcount/20/3600;
  m=(loopcount%(3600*20))/20/60;
  s=(loopcount%(60*20))/20;
  rt_printk("REALTIMEWAVE: Performance:\n  mindelayns==%d\n  maxdelayns==%d\n  errorcount==%d\n  latecount==%d\n  loopcount==%dk (%d:%02d:%02d h:mm:ss)\n",mindelayns,maxdelayns,errorcount,latecount,loopcount,h,m,s);
  rt_printk("REALTIMEWAVE: Cleanup_module() finished\n");
}

void task_waveout_run(int param)
{
  static int count=0;
  static int prevns=0;
  static int internalloopcount=0;
  int curns;
  lsampl_t sample;
  int r;
  unsigned int h,m,s;
  comedi_insn insn;

  /* Prepare an instruction to write. This will simply output
   * whatever is in 'sample' to the analog output of our DAQ card.
   * Note this was tested with a NI-6035E card. The subdevices and
   * other information might be different for your card!
   */
  insn.insn=INSN_WRITE;
  insn.n=1;
  insn.data=&sample;
  insn.subdev=1;
  insn.chanspec=0;

  /* Loop while the module is not being cleaned up */
  while (!stop)
  {
    /* Check time to check performance. We use an internal loopcount
     * to count 1000's of loops, then increment the 'loopcount'.
     * This makes sure that we can run lots of loops (i.e. more than
     * one or two days) without overflowing the 32 bit integer. */
    internalloopcount++;
    if (internalloopcount==1000)
    {
      loopcount++;
      internalloopcount=0;
      /* Every second (20k loops at 50 microsec intervals), print
       * the performance so we can keep track of it while running.
       */
      if (!(loopcount%20))
      {
        h=loopcount/20/3600;
        m=(loopcount%(3600*20))/20/60;
        s=(loopcount%(60*20))/20;
        rt_printk("RTW Perf: min==%d max==%d err==%d late==%d loop==%dk (%d:%02d:%02d h:mm:ss)\n",mindelayns,maxdelayns,errorcount,latecount,loopcount,h,m,s);
      }
    }
    /* Update performance: check time difference between this and
     * previous loop (if there was a previous loop, otherwise prevns
     * equals zero and this is the first loop. Note this means that
     * if the time in nanoseconds overflows an integer and ends up
     * being zero, that loop will not be checked. However, this is
     * a 1 in a billion chance, so don't worry too much about that.
     * (Since we use only time differences the overflowing won't
     * matter to us. Also, in case one of the loops is missed due
     * to the time becoming exactly '0', the next one isn't
     * accidentally seen as 'late', so the counts should still be
     * accurate)
     */
    curns=(int)rt_get_time_ns();
    if (prevns!=0)
    {
      mindelayns=min(mindelayns,curns-prevns);
      maxdelayns=max(maxdelayns,curns-prevns);
      /* Late is defined as more than 50% too late. Some lateness or
       * earlyness can not be avoided due to the resolution of the timer.
       * An addition here could be to check for earlyness as well, and
       * confine the late/early criteria more (say 10% instead of 50%)
       */
      if (curns-prevns>PERIOD*3/2)
        latecount++;
    }
    prevns=curns;

    /* Create a sample to output. Since we can't normally access
     * sine functions from the kernel we will use a simple square
     * wave. Note that depending on the range of your DAQ card,
     * you might want to change the function to accommodate the
     * high and low values. On my NI-6035E the default is -10 to
     * +10 volts for 0..4095, so use values around 2047 to end up
     * with a square wave around 0V. (Use the comedi_calibrate
     * program to make sure these values are accurate)
     */
    //sample=2047+sin((float)count++*6.2832/1000.0);
    sample=2037+20*(count++%2);

    /* Output our sample and check success */
    r=comedi_do_insn(daq,&insn);
    if (r!=1)
      errorcount++;

    /* Note: comedi_data_write() will not work here. I don't know
     * why, but comedi_do_insn will work fine instead...
     */
    /*
    r=comedi_data_write(daq,1,0,0,AREF_GROUND,sample);
    if (r)
      rt_printk("Error: %d\n",r);
    */

    /* And now simply wait until we are allowed to run again */
    rt_task_wait_period();
  }

  rt_printk("REALTIMEWAVE: Stopping thread\n");
}
