/*
    commat_util
    Copyright (C) 2002 Timothy E. Holy <holy@pcg.wustl.edu>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation, version 2.1
    of the License.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
    USA.
*/

#include "mex.h"
#include <comedilib.h>
#include <comedipersist.h>
#include <string.h>
#include "commat.h"

int warnMsg = 1;

const char *cmdfields[] = {
  "subdev",
  "flags",
  "start_src",
  "start_arg",
  "scan_begin_src",
  "scan_begin_arg",
  "convert_src",
  "convert_arg",
  "scan_end_src",
  "scan_end_arg",
  "stop_src",
  "stop_arg",
  "packedchanlist",
  "chanlist",
  "rangelist",
  "areflist",
};
const int n_cmdfields = 16;

const char *arefnames[] = {
  "AREF_GROUND",
  "AREF_COMMON",
  "AREF_DIFF",
  "AREF_OTHER",
};
const unsigned int arefvals[] = {
  AREF_GROUND,
  AREF_COMMON,
  AREF_DIFF,
  AREF_OTHER,
};
const int n_aref = 4;

const char *unitnames[] = {
  "UNIT_volt",
  "UNIT_mA",
  "UNIT_none",
};
const unsigned int unitvals[] = {
  UNIT_volt,
  UNIT_mA,
  UNIT_none,
};
const int n_unit = 3;

const char *srcnames[] = {
  "TRIG_NONE",
  "TRIG_NOW",
  "TRIG_FOLLOW",
  "TRIG_TIME",
  "TRIG_TIMER",
  "TRIG_COUNT",
  "TRIG_EXT",
  "TRIG_INT",
  "TRIG_OTHER",
};
const unsigned int srcvals[] = {
  TRIG_NONE,
  TRIG_NOW,
  TRIG_FOLLOW,
  TRIG_TIME,
  TRIG_TIMER,
  TRIG_COUNT,
  TRIG_EXT,
  TRIG_INT,
  TRIG_OTHER,
};
const int n_src = 9;

const char *subdevnames[] = {
  "COMEDI_SUBD_UNUSED",
  "COMEDI_SUBD_AI",
  "COMEDI_SUBD_AO",
  "COMEDI_SUBD_DI",
  "COMEDI_SUBD_DO",
  "COMEDI_SUBD_DIO",
  "COMEDI_SUBD_COUNTER",
  "COMEDI_SUBD_TIMER",
  "COMEDI_SUBD_MEMORY",
  "COMEDI_SUBD_CALIB",
  "COMEDI_SUBD_PROC"};
const unsigned int subdevvals[] = {
  COMEDI_SUBD_UNUSED,
  COMEDI_SUBD_AI,
  COMEDI_SUBD_AO,
  COMEDI_SUBD_DI,
  COMEDI_SUBD_DO,
  COMEDI_SUBD_DIO,
  COMEDI_SUBD_COUNTER,
  COMEDI_SUBD_TIMER,
  COMEDI_SUBD_MEMORY,
  COMEDI_SUBD_CALIB,
  COMEDI_SUBD_PROC};
const int n_subdevnames = 10;

const char *dio_direction_names[] = {
  "COMEDI_INPUT",
  "COMEDI_OUTPUT"};
const unsigned int dio_direction_vals[] = {
  COMEDI_INPUT,
  COMEDI_OUTPUT};
const int n_dio_directions = 2;

/*
void comedipersist_error(const char *msg)
{
  mexErrMsgTxt(msg);
}
*/

/*
FILE* stderrstream = NULL;

void capture_stderr()
{
  if (stderrstream != NULL)
    mexErrMsgTxt("capture_stderr: stderr is already being captured");
  stderrstream = freopen("stderr.out","w, type=memory", stderr);
  if (stderrstream == NULL) {
    perror("capture_stderr");
    mexErrMsgTxt("capture_stderr: error capturing stderr");
  }
}

char* stderrcpy(char *dest)
{
  int filelength;
  char *cpyresult;

  if (stderrstream == NULL)
    mexErrMsgTxt("stderrcpy: stderr is not being captured");
  if (fclose(stderrstream))
    mexErrMsgTxt("stderrcpy: error closing capture stream");
  stderrstream = fopen("stderr.out","r");
  if (stderrstream == NULL)
    mexErrMsgTxt("stderrcpy: error reopening capture file in read mode");
  if (fseek(stderrstream,0,SEEK_END))
    mexErrMsgTxt("stderrcpy: error seeking end of capture stream");
  filelength = ftell(stderrstream);
  rewind(stderrstream);
  dest = (char *) mxCalloc(filelength, sizeof(char));
  if (!dest)
    mexErrMsgTxt("stderrcpy: error allocating string");
  cpyresult = fgets(dest,filelength,stderrstream);
  if (cpyresult == NULL)
    mexErrMsgTxt("stderrcpy: error copying stream");
  if (fclose(stderrstream))
    mexErrMsgTxt("stderrcpy: error closing capture stream after read");
  stderrstream = NULL;
  return dest;
}
*/

int matstr_len(const mxArray *matstr)
{
  int nrows,ncols;

  if (!mxIsChar(matstr))
    mexErrMsgTxt("matstr_len: argument should be a string");
  nrows = mxGetM(matstr);
  ncols = mxGetN(matstr);
  if (nrows != 1 && ncols != 1)
    mexErrMsgTxt("matstr_len: argument must be a string vector, not matrix");

  return nrows*ncols*sizeof(mxChar) + 1;
}

char *matstr_to_cstr(const mxArray *matstr,char *cstr,int buflen)
{
  if (cstr == NULL)
    mexErrMsgTxt("matstr_to_cstr: error allocating space for c string");
  if (mxGetString(matstr,cstr,buflen))
    mexErrMsgTxt("matstr_to_cstr: some error copying device name");
  return cstr;
}

bool IsScalar(const mxArray *m)
{
  return (mxGetNumberOfElements(m) == 1);
}

int intmatch(unsigned int uint,const unsigned int vals[],int n_vallist)
{
  int i;

  i = 0;
  while (i < n_vallist)
    if (uint == vals[i])
      break;
    else
      i++;
  if (i < n_vallist)
    return i;
  else
    return -1;
}
      
int strmatch(const char *str,const char *strlist[],int n_strlist)
{
  int i;

  i = 0;
  while (i < n_strlist)
    if (!strcmp(str,strlist[i]))
      break;
    else
      i++;
  if (i < n_strlist)
    return i;
  else {
    if (warnMsg) {
      mexPrintf("%s is not among the following choices:\n",str);
      for (i = 0; i < n_strlist; i++)
	mexPrintf("  %s\n",strlist[i]);
    }
    return -1;
  }
}


int getvalfromstr(const mxArray *matstr,const char *strs[],int nvals)
{
  int buflen;
  char *cstr;
  int ret;

  buflen = matstr_len(matstr);
  cstr = (char*) mxCalloc(buflen,sizeof(char));
  if (!cstr)
    mexErrMsgTxt("getvalfromstr: error allocating space for string");
  matstr_to_cstr(matstr,cstr,buflen);
  ret = strmatch(cstr,strs,nvals);
  mxFree(cstr);
  if (ret < 0)
    mexErrMsgTxt("getvalfromstr: string is not in list");
  return ret;
}

comedi_t* get_comedi_device_safe(const char *devname)
{
  if (!comedi_device_isopen(devname)) {
    mexPrintf("Error using device %s:\n",devname);
    mexErrMsgTxt("Device is not open");
  }
  return get_comedi_device(devname);
}
			    

void matstruct_to_cmd(const mxArray *matcmd,comedi_cmd *comcmd)
{
  int nfields,i;
  const mxArray *curarg;
  const char *curfieldname;
  int curfieldindx;
  unsigned int *chanlist,*rangelist,*areflist,*packedchanlist;
  int nchans_chan,nchans_range,nchans_aref,nchans_packed;
  double *chanp;
  int j;

  if (!mxIsStruct(matcmd))
    mexErrMsgTxt("Error matstruct_to_cmd: first input must be a MATLAB structure");
  if (mxGetNumberOfElements(matcmd) != 1)
    mexErrMsgTxt("Error matstruct_to_cmd: first input must be have only 1 structure element");

  memset(comcmd,0,sizeof(*comcmd));
  nchans_chan = nchans_range = nchans_aref = nchans_packed = 0;

  nfields = mxGetNumberOfFields(matcmd);

  for (i = 0; i < nfields; i++) {
    curfieldname = mxGetFieldNameByNumber(matcmd,i);
    curfieldindx = strmatch(curfieldname, cmdfields, n_cmdfields);
    switch (curfieldindx) {
    case -1:
      mexPrintf("Fieldname %s not recognized\n",curfieldname);
      mexErrMsgTxt("matstruct_to_cmd");
      break;
    case 0:   // subdev
      comcmd->subdev = (unsigned int) mxGetScalar(mxGetFieldByNumber(matcmd,0,i));
      break;
    case 1:   // flags
      if (!IsScalar(mxGetFieldByNumber(matcmd,0,i)))
	mexErrMsgTxt("nonscalar flags not yet implemented");
      comcmd->flags = (unsigned int) mxGetScalar(mxGetFieldByNumber(matcmd,0,i));
      break;
    case 2:   // start_src
      comcmd->start_src = srcvals[getvalfromstr(mxGetFieldByNumber(matcmd,0,i),srcnames,n_src)];
      break;
    case 3:   // start_arg
      comcmd->start_arg = (unsigned int) mxGetScalar(mxGetFieldByNumber(matcmd,0,i));
      break;
    case 4:   // scan_begin_src
      comcmd->scan_begin_src = srcvals[getvalfromstr(mxGetFieldByNumber(matcmd,0,i),srcnames,n_src)];
      break;
    case 5:   // scan_begin_arg
      comcmd->scan_begin_arg = (unsigned int) mxGetScalar(mxGetFieldByNumber(matcmd,0,i));
      break;
    case 6:   // convert_src
      comcmd->convert_src = srcvals[getvalfromstr(mxGetFieldByNumber(matcmd,0,i),srcnames,n_src)];
      break;
    case 7:   // convert_arg
      comcmd->convert_arg = (unsigned int) mxGetScalar(mxGetFieldByNumber(matcmd,0,i));
      break;
    case 8:   // scan_end_src
      comcmd->scan_end_src = srcvals[getvalfromstr(mxGetFieldByNumber(matcmd,0,i),srcnames,n_src)];
      break;
    case 9:   // scan_end_arg
      comcmd->scan_end_arg = (unsigned int) mxGetScalar(mxGetFieldByNumber(matcmd,0,i));
      break;
    case 10:  // scan_src
      comcmd->stop_src = srcvals[getvalfromstr(mxGetFieldByNumber(matcmd,0,i),srcnames,n_src)];
      break;
    case 11:  // stop_arg
      comcmd->stop_arg = (unsigned int) mxGetScalar(mxGetFieldByNumber(matcmd,0,i));
      break;
    case 12:  // packedchanlist
      curarg = mxGetFieldByNumber(matcmd,0,i);
      nchans_packed = mxGetNumberOfElements(curarg);
      packedchanlist = (unsigned int *) mxCalloc(nchans_packed,sizeof(unsigned int));
      if (!packedchanlist)
	mexErrMsgTxt("matstruct_to_cmd: error allocating space for packedchanlist");
      chanp = mxGetPr(curarg);
      for (j = 0; j < nchans_packed; j++)
	packedchanlist[j] = (unsigned int) chanp[j];
      break;
    case 13:  // chanlist
      curarg = mxGetFieldByNumber(matcmd,0,i);
      nchans_chan = mxGetNumberOfElements(curarg);
      chanlist = (unsigned int *) mxCalloc(nchans_chan,sizeof(unsigned int));
      if (!chanlist)
	mexErrMsgTxt("matstruct_to_cmd: error allocating space for chanlist");
      chanp = mxGetPr(curarg);
      for (j = 0; j < nchans_chan; j++)
	chanlist[j] = (unsigned int) chanp[j];
      break;
    case 14:  // rangelist
      curarg = mxGetFieldByNumber(matcmd,0,i);
      nchans_range = mxGetNumberOfElements(curarg);
      rangelist = (unsigned int *) mxCalloc(nchans_range,sizeof(unsigned int));
      if (!rangelist)
	mexErrMsgTxt("matstruct_to_cmd: error allocating space for rangelist");
      chanp = mxGetPr(curarg);
      for (j = 0; j < nchans_range; j++)
	rangelist[j] = (unsigned int) chanp[j];
      break;
    case 15:  // areflist
      curarg = mxGetFieldByNumber(matcmd,0,i);
      nchans_aref = mxGetNumberOfElements(curarg);
      areflist = (unsigned int *) mxCalloc(nchans_aref,sizeof(unsigned int));
      if (!areflist)
	mexErrMsgTxt("matstruct_to_cmd: error allocating space for areflist");
      chanp = mxGetPr(curarg);
      for (j = 0; j < nchans_aref; j++)
	areflist[j] = arefvals[(int) chanp[j]];
      break;
    }
  }
  // Check to see if channels are defined; if so, do CR_PACK
  if (nchans_chan && nchans_packed == 0) {
    if (!nchans_range) {
      rangelist = (unsigned int *) mxCalloc(nchans_chan,sizeof(unsigned int));
      if (!rangelist)
	mexErrMsgTxt("matstruct_to_cmd: error allocating space for rangelist");
      for (j = 0; j < nchans_chan; j++)
	rangelist[j] = 0;
    }
    else if (nchans_range != nchans_chan)
      mexErrMsgTxt("matstr_to_cmd: the number of channels and ranges do not match");
    if (!nchans_aref) {
      areflist = (unsigned int *) mxCalloc(nchans_chan,sizeof(unsigned int));
      if (!areflist)
	mexErrMsgTxt("matstruct_to_cmd: error allocating space for areflist");
      for (j = 0; j < nchans_chan; j++)
	areflist[j] = AREF_GROUND;
    }
    else if (nchans_aref != nchans_chan)
      mexErrMsgTxt("matstr_to_cmd: the number of channels and arefs do not match");
    packedchanlist = (unsigned int *) mxCalloc(nchans_chan,sizeof(unsigned int));
    if (!packedchanlist)
      mexErrMsgTxt("matstruct_to_cmd: error allocating space for packedchanlist");
    for (j = 0; j < nchans_chan; j++) {
      packedchanlist[j] = CR_PACK(chanlist[j],rangelist[j],areflist[j]);
    }
    mxFree(chanlist);
    mxFree(rangelist);
    mxFree(areflist);
    comcmd->chanlist = packedchanlist;
    comcmd->chanlist_len = nchans_chan;
  }
  else {
    if (nchans_chan)
      mexErrMsgTxt("matstr_to_cmd: Can't have both chanlist and packedchanlist");
    comcmd->chanlist = packedchanlist;
    comcmd->chanlist_len = nchans_packed;
  }
}

void cmd_to_matstruct(const comedi_cmd *comcmd, mxArray *matcmd)
{
  mxArray *packedchanlist;
  double *pchan;
  int j;

  mxSetField(matcmd,0,"subdev",mxCreateScalarDouble(comcmd->subdev));
  mxSetField(matcmd,0,"flags",mxCreateScalarDouble(comcmd->flags));
  mxSetField(matcmd,0,"start_src",mxCreateString(srcnames[intmatch(comcmd->start_src,srcvals,n_src)]));
  mxSetField(matcmd,0,"start_arg",mxCreateScalarDouble(comcmd->start_arg));
  mxSetField(matcmd,0,"scan_begin_src",mxCreateString(srcnames[intmatch(comcmd->scan_begin_src,srcvals,n_src)]));
  mxSetField(matcmd,0,"scan_begin_arg",mxCreateScalarDouble(comcmd->scan_begin_arg));
  mxSetField(matcmd,0,"convert_src",mxCreateString(srcnames[intmatch(comcmd->convert_src,srcvals,n_src)]));
  mxSetField(matcmd,0,"convert_arg",mxCreateScalarDouble(comcmd->convert_arg));
  mxSetField(matcmd,0,"scan_end_src",mxCreateString(srcnames[intmatch(comcmd->scan_end_src,srcvals,n_src)]));
  mxSetField(matcmd,0,"scan_end_arg",mxCreateScalarDouble(comcmd->scan_end_arg));
  mxSetField(matcmd,0,"stop_src",mxCreateString(srcnames[intmatch(comcmd->stop_src,srcvals,n_src)]));
  mxSetField(matcmd,0,"stop_arg",mxCreateScalarDouble(comcmd->stop_arg));
  packedchanlist = mxCreateDoubleMatrix((int) comcmd->chanlist_len,1,mxREAL);
  pchan = mxGetPr(packedchanlist);
  for (j = 0; j < (int) comcmd->chanlist_len; j++) {
    pchan[j] = comcmd->chanlist[j];
  }
  mxSetField(matcmd,0,"packedchanlist",packedchanlist);
}
	
void comedi_cmd_mxFree(comedi_cmd *comcmd)
{
  mxFree(comcmd->chanlist);
}

/* 
 * The following output functions were copied directly from
 * David A. Schleef's common.c file in the comedilib demo directory.
 */

char *cmd_src(int src,char *buf)
{
        buf[0]=0;
 
        if(src&TRIG_NONE)strcat(buf,"none|");
        if(src&TRIG_NOW)strcat(buf,"now|");
        if(src&TRIG_FOLLOW)strcat(buf, "follow|");
        if(src&TRIG_TIME)strcat(buf, "time|");
        if(src&TRIG_TIMER)strcat(buf, "timer|");
        if(src&TRIG_COUNT)strcat(buf, "count|");
        if(src&TRIG_EXT)strcat(buf, "ext|");
        if(src&TRIG_INT)strcat(buf, "int|");
#ifdef TRIG_OTHER
        if(src&TRIG_OTHER)strcat(buf, "other|");
#endif
 
        if(strlen(buf)==0){
                sprintf(buf,"unknown(0x%08x)",src);
        }else{
                buf[strlen(buf)-1]=0;
        }
 
        return buf;
}
 
void dump_cmd(FILE *out,comedi_cmd *cmd)
{
        char buf[100];
 
        fprintf(out,"start:      %-8s %d\n",
                cmd_src(cmd->start_src,buf),
                cmd->start_arg);
 
        fprintf(out,"scan_begin: %-8s %d\n",
                cmd_src(cmd->scan_begin_src,buf),
                cmd->scan_begin_arg);
 
        fprintf(out,"convert:    %-8s %d\n",
                cmd_src(cmd->convert_src,buf),
                cmd->convert_arg);
 
        fprintf(out,"scan_end:   %-8s %d\n",
                cmd_src(cmd->scan_end_src,buf),
                cmd->scan_end_arg);
 
        fprintf(out,"stop:       %-8s %d\n",
                cmd_src(cmd->stop_src,buf),
                cmd->stop_arg);
}
