/* whatami:....daq_test.c
   why:........To test DAQ board operation
   ally:.......daq_test.h
   requi:......rtlinux, comedi

   totry:......compile, insert, check dmesg for errors
   apply:......AnalogOut_0 should mimic AnalogIn_0 

   comply:.....GNU General Public License
   supply:.....Freely to all kind souls, the world over!

   by:.........Chuck Dorval
   reply:......bionic@bu.edu
   good-bye:...July 15, 2001
*/

//#define COMEDI_CMD_STYLE

// ***** System Headers ***** //
#include <rtl.h>
#include <pthread.h>

#include <linux/comedi.h>
#include <linux/comedilib.h>


// ***** DAQ Board Constants ***** //
#define SAMPLE_PERIOD    500000    // Inter-sample period in ns
#define SAMPLE_DURATION   10000    // Single sample duration in ns

#define DAQ_DEV_NAME  "/dev/comedi0"    // DAQ board device number

#define AI_SUBD            0       // Analog Input subdevice number
#define AI_CHAN            0       // Analog Input channel
#define AI_GAIN            0       // Analog Input range parameter
#define AI_REF     AREF_DIFF       // Analog Input lead style
#define AI_MAX         65536       // Analog Input max value 2^16

#define AO_SUBD            1       // Analog Output subdevice number
#define AO_CHAN            0       // Analog Output channel
#define AO_GAIN            0       // Analog Output range parameter
#define AO_REF   AREF_GROUND       // Analog Output lead style
#define AO_MAX          4096       // Analog Output max value 2^12


// ***** Function Declarations *****/
#ifdef COMEDI_CMD_STYLE
int   callback(unsigned int, void*);   // Comedi Callback function
#endif 

int   init_daq_channels(void);         // Initialize DAQ channels
void *daq_code(void*);                 // DAQ Periodic function


// ***** Global Variables ***** //

// RT required 
pthread_t     daq_thread;            // main rtlinux thread

// DAQ Board Required External Variables; some from comedi_config
comedi_t     *daq_dev;               // DAQ device id

#ifdef COMEDI_CMD_STYLE
comedi_cmd    ai_cmd;                // main AI comedi command
sampl_t       ai_data;               // comedi AI data points
#else
comedi_insn   ai_insn;               // main AI comedi instruction
lsampl_t      ai_data;               // comedi AI data point
#endif
unsigned int  ai_chanspec;           // packed comedi AI channel

comedi_insn   ao_insn;               // main AO comedi instruction
lsampl_t      ao_data;               // comedi AO data point
unsigned int  ao_chanspec;           // packed comedi AO channel


// ***** Operational Functions (2) ***** //

#ifdef COMEDI_CMD_STLYE
// Function called by DAQ trigger
int callback(unsigned int flags, void *arg)
{
  pthread_wakeup_np( daq_thread );
  return 1;
}
#endif

// Task reads/writes data to DAQ board
void *daq_code(void *param)
{  
  static short ai_signed, ao_signed;
  static short ai_zero,   ao_zero,   io_ratio;

  ai_zero  = AI_MAX/2;        // middle AI value corresponds to zero
  ao_zero  = AO_MAX/2;        // middle AO value corresponds to zero
  io_ratio = AI_MAX/AO_MAX;   // ratio of ai to ao values

  while (1)                               // periodic task, must loop
    {
      pthread_wait_np();      // Wait for DAQ signal

#ifndef COMEDI_CMD_STYLE
      // Read analog in from DAQ
      if ( comedi_do_insn(daq_dev,&ai_insn) != 1 )
	rtl_printf("Cannot read input value from ADC\n");
#endif

      // Convert bit count from input to output
      ai_signed = (short)ai_data - ai_zero;
      ao_signed = ai_signed / io_ratio;
      ao_data   = (unsigned int)(ao_signed + ao_zero);

      // Write analog out to DAQ
      if ( comedi_do_insn(daq_dev,&ao_insn) != 1 )
	rtl_printf( "Cannot write ouput value to DAC\n" );
    }    
}


// ***** Initialization Functions (1) ***** //

// Board initialization - very sensitive to errors
int init_daq_channels(void)
{
  int error_flag=0;
  char daq_dev_name[12];

  // initialize structures
#ifdef COMEDI_CMD_STYLE
  memset(&ai_cmd, 0,sizeof(ai_cmd));
#else
  memset(&ai_insn,0,sizeof(ai_insn));
#endif
  memset(&ao_insn,0,sizeof(ao_insn));

  // Pack analog channel values for Comedi convenience
  ai_chanspec  = CR_PACK(AI_CHAN,AI_GAIN,AI_REF);
  ao_chanspec  = CR_PACK(AO_CHAN,AO_GAIN,AO_REF);

#ifdef COMEDI_CMD_STYLE
  // Set up AI cmd structure
  ai_cmd.chanlist     = &ai_chanspec;      // stores vales packed above
  ai_cmd.chanlist_len = 1;                 // number of AI channels
  ai_cmd.data         = &ai_data;          // where to put read data
  ai_cmd.data_len     = sizeof(sampl_t);   // size of read data

  ai_cmd.subdev       = AI_SUBD;        // ai subdevice number
  ai_cmd.flags        = 0;              // no special flags
  ai_cmd.start_src    = TRIG_NOW;       // start cmd timer on fxn call
  ai_cmd.start_arg    = 0;              // start cmd 0ns after fxn call

  ai_cmd.scan_begin_src = TRIG_TIMER;       // use on board clock
  ai_cmd.scan_begin_arg = SAMPLE_PERIOD;    // time between samples
  ai_cmd.convert_src    = TRIG_ANY;         // always sample channel
  ai_cmd.convert_arg    = SAMPLE_DURATION;  // time of each sample

  ai_cmd.scan_end_src   = TRIG_COUNT;       // scan once per channel
  ai_cmd.scan_end_arg   = 1;                // number of channels
  ai_cmd.stop_src       = TRIG_NONE;        // continuous AI scanning
  ai_cmd.stop_arg       = 0;                // unused with TRIG_NONE

#else
  // Set up AI insn structure
  ai_insn.insn   = INSN_READ;      // This instruction is for reading
  ai_insn.subdev = AI_SUBD;        // ai subdevice number
  ai_insn.n      = 1;              // number of data points to read

  ai_insn.data     = &ai_data;     // data value to read from AI
  ai_insn.chanspec = ai_chanspec;  // packed values from above
#endif

  // Set up AO insn structure
  ao_insn.insn   = INSN_WRITE;     // This instruction is for writing
  ao_insn.subdev = AO_SUBD;        // ao subdevice number
  ao_insn.n      = 1;              // number of data points to write

  ao_insn.data     = &ao_data;     // data value to write to AO
  ao_insn.chanspec = ao_chanspec;  // packed values from above


  // open comedi device
  sprintf(daq_dev_name,DAQ_DEV_NAME);
  if ((daq_dev  = comedi_open(daq_dev_name))==NULL)
    {
      printk("Error. Cannot open comedi device.\n");
      return(-1);
    }

  // lock subdevices
  if ( !(error_flag = comedi_lock(daq_dev,AI_SUBD)) )
  if ( !(error_flag = comedi_lock(daq_dev,AO_SUBD)) )
  
#ifdef COMEDI_CMD_STYLE
  // register callback function
  error_flag = comedi_register_callback
      (daq_dev, AI_SUBD, COMEDI_CB_EOS, &callback, (void *)0)
#endif 
    ;

  return(error_flag);
}


// ***** Module Standard Functions (2) ***** //

int init_module(void)
{
  pthread_attr_t attr;           // p_thread structure

  // initialize daq board A/D & D/A channels
  if ( init_daq_channels() )
    printk( "Cannot establish AIn or AOut channels\n" );

  // initialize pthread attributes
  if(  pthread_attr_init(&attr) ) 
    printk( "Cannot initialize pthread attributes\n" );

  // initialize the main thread
  if( pthread_create(&daq_thread, &attr, daq_code,(void *)1) )
    printk( "Cannot create pthread\n" );

#ifdef COMEDI_CMD_STYLE
  // start data acquisition
  if ( comedi_command(daq_dev,&ai_cmd) )
    printk( "Comedi command not correctly implemented\n" );
#else
  if (pthread_make_periodic_np
                   (daq_thread,clock_gethrtime(CLOCK_RTL_SCHED),SAMPLE_PERIOD))
      printk( "Failed to make pthread periodic.\n");
#endif

  return 0;
}


void cleanup_module(void)
{
#ifdef COMEDI_CMD_STYLE
  // Inactivate comedi triggering mechanism
  if ( comedi_cancel(daq_dev,AI_SUBD) )
    printk( "Comedi trigger not inactivated\n" );
#endif

  // Unlock AI subdevice so other comedi programs can use
  if ( comedi_unlock(daq_dev,AI_SUBD) )
    printk( "AI channel(s) remain inappropriately locked\n" );

  // Unlock AO subdevice so other comedi programs can use
  if ( comedi_unlock(daq_dev,AO_SUBD) )
    printk( "AO channel(s) remain inappropriately locked\n" );

  // close comedi device, no return value yet
  if ( comedi_close(daq_dev) )
    printk( "Error in comedi_close.\n");

  // Destroy periodic thread
  if ( pthread_delete_np(daq_thread) )
    printk( "Thread not destroyed\n" );
}










