/*
    comedi/drivers/ni_660x.c
    Hardware driver for NI 660x devices

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
   Driver: ni_660x.o
   Description: National Instruments 660x counter/timer boards
   Devices:
    (National Instruments) PCI-6601 [ni_660x]
	   Four 32 bits counters/timers + 32 digital IO (24 of them
	   shared with counters) + 1 channel DMA + interrupts
    (National Instruments) PCI-6602 [ni_660x]
	   eight 32 bits counters/timers + 32 digital IO (24 of them
	   shared with counters) + 3 channels DMA + interrupts
	  Both devices contain the MITE PCI interface.
   Author: 
     J.P. Mellor <jpmellor@rose-hulman.edu>: original driver
	  Herman.Bruyninckx@mech.kuleuven.ac.be,
	  Wim.Meeussen@mech.kuleuven.ac.be,
	  Klaas.Gadeyne@mech.kuleuven.ac.be: extended with 6602,
	  completely rewrote GPCT subdevice-driver
   Updated: Tue, 15 Apr 2003 10:57:22 +0200
   Status: experimental
	 
   Encoders work, but only with instructions, commands are not
	supported yet. DIO doesn't work yet. Interrupts do not
	work. 

	Since there is not a well-defined API for the use of counters
	in comedi yet, we use the one suggested by Herman Bruyninckx
	Another possibility is the API suggested by DS in  
	comedi/Documentation/comedi/counter-spec.  
	
   References:
	   DAQ 660x Register-Level Programmer Manual  (NI 370505A-01)
	   DAQ 6601/6602 User Manual (NI 322137B-01)

   Things to do:
   - Add DMA support (see mite.c and ni_pcidio.c for examples)
   - Add commands (copy from ni_pcidio.c ?)
   - Add interrupts
   - Extend "Application possibilities" for the GPCT subdevice (eg. Time
   Measurement, ...)
*/

#include <linux/comedidev.h>
#include "mite.h"
#include "ni_660x.h" // Some constants necessary for the Application
		     // Programmer not yet accepted in <comedidev.h>

#define CTRS_PER_CHIP 4 // The number of counters per ni-tio chip
#define DATA_1B 0x1 // 1 byte = 8 bits data
#define DATA_2B 0x2 // 2 bytes = 16 bit data
#define DATA_4B 0x4 // 4 bytes = 32 bit data

/* See Register-Level Programmer Manual page 3.1 */
typedef enum 
  {
    G0InterruptAcknowledge,
    G0StatusRegister,
    G1InterruptAcknowledge,
    G1StatusRegister,
    G01StatusRegister,
    G0CommandRegister,
    G1CommandRegister,
    G0HWSaveRegister,
    G1HWSaveRegister,
    G0SWSaveRegister,
    G1SWSaveRegister,
    G0ModeRegister,
    G01JointStatus1Register,
    G1ModeRegister,
    G0LoadARegister,
    G01JointStatus2Register,
    G0LoadBRegister,
    G1LoadARegister,
    G1LoadBRegister,
    G0InputSelectRegister,
    G1InputSelectRegister,
    G01JointResetRegister,
    G0InterruptEnable,
    G1InterruptEnable,
    G0CountingModeRegister,
    G1CountingModeRegister,
    G0SecondGateRegister,
    G1SecondGateRegister,
    G0DMAConfigRegister,
    G0DMAStatusRegister,
    G1DMAConfigRegister,
    G1DMAStatusRegister,
    G2InterruptAcknowledge,
    G2StatusRegister,
    G3InterruptAcknowledge,
    G3StatusRegister,
    G23StatusRegister,
    G2CommandRegister,
    G3CommandRegister,
    G2HWSaveRegister,
    G3HWSaveRegister,
    G2SWSaveRegister,
    G3SWSaveRegister,
    G2ModeRegister,
    G23JointStatus1Register,
    G3ModeRegister,
    G2LoadARegister,
    G23JointStatus2Register,
    G2LoadBRegister,
    G3LoadARegister,
    G3LoadBRegister,
    G2InputSelectRegister,
    G3InputSelectRegister,
    G23JointResetRegister,
    G2InterruptEnable,
    G3InterruptEnable,
    G2CountingModeRegister,
    G3CountingModeRegister,
    G3SecondGateRegister,
    G2SecondGateRegister,
    G2DMAConfigRegister,
    G2DMAStatusRegister,
    G3DMAConfigRegister,
    G3DMAStatusRegister,
    ClockConfigRegister,
    NumRegisters, 
  } NI_660xRegisters;

typedef struct 
{
  char *name; // Register Name
  int offset; // Offset from base address from GPCT chip
  int direction; // read or write, ie INSN_READ or ...
  int size; // 1 byte, 2 bytes, or 4 bytes
} NI_660xRegisterData;

const NI_660xRegisterData registerData[NumRegisters] = 
  {
    {"G0 Interrupt Acknowledge", 0x004, INSN_WRITE, DATA_2B},
    {"G0 Status Register", 0x004, INSN_READ, DATA_2B},
    {"G1 Interrupt Acknowledge", 0x006, INSN_WRITE, DATA_2B},
    {"G1 Status Register", 0x006, INSN_READ, DATA_2B},
    {"G01 Status Register ", 0x008, INSN_READ, DATA_2B},
    {"G0 Command Register", 0x00C, INSN_WRITE, DATA_2B},
    {"G1 Command Register", 0x00E, INSN_WRITE, DATA_2B},
    {"G0 HW Save Register", 0x010, INSN_READ, DATA_4B},
    {"G1 HW Save Register", 0x014, INSN_READ, DATA_4B},
    {"G0 SW Save Register", 0x018, INSN_READ, DATA_4B},
    {"G1 SW Save Register", 0x01C, INSN_READ, DATA_4B},
    {"G0 Mode Register", 0x034, INSN_WRITE, DATA_2B},
    {"G01 Joint Status 1 Register", 0x036, INSN_READ, DATA_2B},
    {"G1 Mode Register", 0x036, INSN_WRITE, DATA_2B},
    {"G0 Load A Register", 0x038, INSN_WRITE, DATA_4B},
    {"G01 Joint Status 2 Register", 0x03A, INSN_READ, DATA_2B},
    {"G0 Load B Register", 0x03C, INSN_WRITE, DATA_4B},
    {"G1 Load A Register", 0x040, INSN_WRITE, DATA_4B},
    {"G1 Load B Register", 0x044, INSN_WRITE, DATA_4B},
    {"G0 Input Select Register", 0x048, INSN_WRITE, DATA_2B},
    {"G1 Input Select Register", 0x04A, INSN_WRITE, DATA_2B},
    {"G01 Joint Reset Register", 0x090, INSN_WRITE, DATA_2B},
    {"G0 Interrupt Enable", 0x092, INSN_WRITE, DATA_2B},
    {"G1 Interrupt Enable", 0x096, INSN_WRITE, DATA_2B},
    {"G0 Counting Mode Register", 0x0B0, INSN_WRITE, DATA_2B},
    {"G1 Counting Mode Register", 0x0B2, INSN_WRITE, DATA_2B},
    {"G0 Second Gate Register", 0x0B4, INSN_WRITE, DATA_2B},
    {"G1 Second Gate Register", 0x0B6, INSN_WRITE, DATA_2B},
    {"G0 DMA Config Register", 0x0B8, INSN_WRITE, DATA_2B},
    {"G0 DMA Status Register", 0x0B8, INSN_READ, DATA_2B},
    {"G1 DMA Config Register", 0x0BA, INSN_WRITE, DATA_2B},
    {"G1 DMA Status Register", 0x0BA, INSN_READ, DATA_2B},
    {"G2 Interrupt Acknowledge", 0x104, INSN_WRITE, DATA_2B},
    {"G2 Status Register", 0x104, INSN_READ, DATA_2B},
    {"G3 Interrupt Acknowledge", 0x106, INSN_WRITE, DATA_2B},
    {"G3 Status Register", 0x106, INSN_READ, DATA_2B},
    {"G23 Status Register", 0x108, INSN_READ, DATA_2B},
    {"G2 Command Register", 0x10C, INSN_WRITE, DATA_2B},
    {"G3 Command Register", 0x10E, INSN_WRITE, DATA_2B},
    {"G2 HW Save Register", 0x110, INSN_READ, DATA_4B},
    {"G3 HW Save Register", 0x114, INSN_READ, DATA_4B},
    {"G2 SW Save Register", 0x118, INSN_READ, DATA_4B},
    {"G3 SW Save Register", 0x11C, INSN_READ, DATA_4B},
    {"G2 Mode Register", 0x134, INSN_WRITE, DATA_2B},
    {"G23 Joint Status 1 Register", 0x136, INSN_READ, DATA_2B},
    {"G3 Mode Register", 0x136, INSN_WRITE, DATA_2B},
    {"G2 Load A Register", 0x138, INSN_WRITE, DATA_4B},
    {"G23 Joint Status 2 Register", 0x13A, INSN_READ, DATA_2B},
    {"G2 Load B Register", 0x13C, INSN_WRITE, DATA_4B},
    {"G3 Load A Register", 0x140, INSN_WRITE, DATA_4B},
    {"G3 Load B Register", 0x144, INSN_WRITE, DATA_4B},
    {"G2 Input Select Register", 0x148, INSN_WRITE, DATA_2B},
    {"G3 Input Select Register", 0x14A, INSN_WRITE, DATA_2B},
    {"G23 Joint Reset Register", 0x190, INSN_WRITE, DATA_2B},
    {"G2 Interrupt Enable", 0x192, INSN_WRITE, DATA_2B},
    {"G3 Interrupt Enable", 0x196, INSN_WRITE, DATA_2B},
    {"G2 Counting Mode Register", 0x1B0, INSN_WRITE, DATA_2B},
    {"G3 Counting Mode Register", 0x1B2, INSN_WRITE, DATA_2B},
    {"G3 Second Gate Register", 0x1B6, INSN_WRITE, DATA_2B},
    {"G2 Second Gate Register", 0x1B4, INSN_WRITE, DATA_2B},

    {"G2 DMA Config Register", 0x1B8, INSN_WRITE, DATA_2B},
    {"G2 DMA Status Register", 0x1B8, INSN_READ, DATA_2B},
    {"G3 DMA Config Register", 0x1BA, INSN_WRITE, DATA_2B},
    {"G3 DMA Status Register", 0x1BA, INSN_READ, DATA_2B},
    {"Clock Config Register", 0x73C, INSN_WRITE, DATA_4B}
  };


/* Different Application Classes for GPCT Subdevices */
/* The list is not exhaustive and needs discussion! */
typedef enum
  {
    CountingAndTimeMeasurement,
    PulseGeneration,
    PositionMeasurement,
    Miscellaneous
  } NI_660x_GPCT_AppClass;


/* Config struct for different GPCT subdevice Application Classes and
   their options 
*/
typedef struct
{
  NI_660x_GPCT_AppClass App;
  int data[6]; /* Application dependent data, eg. for encoders, See
		  mail Herman Bruyninckx TODO (add link to ML archive,
		  but website currently down */
} NI_660x_GPCT_Config;

#define NI_660X_GPCT_MAXCHANNELS 8 // To avoid dyn. mem. allocation
NI_660x_GPCT_Config ni_660x_gpct_config[NI_660X_GPCT_MAXCHANNELS];

/* Some bits to write in different registers */
#define UpDownDown		0x0<<5 // always count down
#define UpDownUp		0x1<<5 // always count up
#define UpDownHardware		0x1<<6 // up/down depending on
				       // hardware pin
#define UpDownGate		0x3<<5 // depending on hardware
				       // internal GATE 
#define Disarm			0x1<<4
#define Load			0x1<<2
#define Arm			0x1<<0

#define IndexPhaseLowLow	0x0<<5 // Index Pulse active when both
				       // A and B are low
#define IndexPhaseLowHigh	0x1<<5 // ...
#define IndexPhaseHighLow	0x2<<5
#define IndexPhaseHighHigh	0x3<<5

#define IndexMode		0x1<<4
// For quadrature encoders
#define CountingModeNormal	0x0<<0
#define CountingModeQuadX1	0x1<<0
#define CountingModeQuadX2	0x2<<0
#define CountingModeQuadX4	0x3<<0

// For 2-pulse encoders
#define CountingModeTwoPulse	0x4<<0
#define CountingModeSynchronous	0x6<<0

#define GateSelectPin38		0x1<<8 // Take internal time-based 20
				       // MHz clockx
#define SourceSelectTimebase1	0x0<<2	
#define TriggerModeStartStop	0x0<<3
#define TriggerModeStopStart	0x1<<3
#define TriggerModeStart	0x2<<3
#define TriggerModeNotUsed	0x3<<3
#define GatingModeDisabled	0x0<<0
#define GatingModeLevel		0x1<<0
#define GatingModeRising	0x2<<0
#define GatingModeFalling	0x3<<0
#define G1Reset			0x1<<3
#define G0Reset			0x1<<2
#define G1Armed			0x1<<9
#define G0Armed			0x1<<8
// kind of ENABLE for the second counter
#define CounterSwap             0x1<<21

// Offset of the GPCT chips from the base-adress of the card
const int GPCT_OFFSET[2] = {0x0,0x800}; /* First chip is at base-adress +
					   0x00, etc. */

/* Board description*/
typedef struct
{
  unsigned short dev_id; /* `lspci` will show you this */
  char *name;
  int n_ctrs; /* total number of counters */
  int cnt_bits; /* number of bits in each counter */
} ni_660x_board;

static ni_660x_board ni_660x_boards[] = 
{
  {
    dev_id       : 0x2c60,
    name         : "PCI-6601",
    n_ctrs       : 1*CTRS_PER_CHIP,
    cnt_bits     : 32,
  },
  {
    dev_id       : 0x1310,
    name         : "PCI-6602",
    n_ctrs       : 2*CTRS_PER_CHIP,
    cnt_bits     : 32,
  },
};

static struct pci_device_id ni_660x_pci_table[] __devinitdata = {
	{ PCI_VENDOR_ID_NATINST, 0x2c60, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
	{ PCI_VENDOR_ID_NATINST, 0x1310, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
	{ 0 }
};
MODULE_DEVICE_TABLE(pci, ni_660x_pci_table);

#define thisboard ((ni_660x_board *)dev->board_ptr)
/* initialized in ni_660x_find_device() */

typedef struct
{
	struct mite_struct *mite;
	int boardtype;
}ni_660x_private;

#define devpriv ((ni_660x_private *)dev->private)
#define n_ni_660x_boards (sizeof(ni_660x_boards)/sizeof(ni_660x_boards[0]))

static int ni_660x_attach(comedi_device *dev,comedi_devconfig *it);
static int ni_660x_detach(comedi_device *dev);

static comedi_driver driver_ni_660x=
{
	driver_name:	"ni_660x",
	module:		THIS_MODULE,
	attach:		ni_660x_attach,
	detach:		ni_660x_detach,
};

COMEDI_INITCLEANUP(driver_ni_660x);

static int ni_660x_find_device(comedi_device *dev,int bus,int slot);

/* Possible instructions for a GPCT */
static int ni_660x_GPCT_rinsn(comedi_device *dev,
			      comedi_subdevice *s,
			      comedi_insn *insn,
			      lsampl_t *data);
static int ni_660x_GPCT_insn_config(comedi_device *dev,
				    comedi_subdevice *s,
				    comedi_insn *insn,
				    lsampl_t *data);
static int ni_660x_GPCT_winsn(comedi_device *dev,
			      comedi_subdevice *s,
			      comedi_insn *insn,
			      lsampl_t * data);

/* Possible instructions for Digital IO: Not implemented yet! */
static int ni_660x_dio_insn_config(comedi_device *dev,
				   comedi_subdevice *s,
				   comedi_insn *insn,
				   lsampl_t *data);
static int ni_660x_dio_insn_bits(comedi_device *dev,
				 comedi_subdevice *s,
				 comedi_insn *insn,
				 lsampl_t *data);


static int ni_660x_attach(comedi_device *dev,comedi_devconfig *it)
{
  comedi_subdevice *s;
  int ret;

  printk("comedi%d: ni_660x: ",dev->minor);

  if ((ret=alloc_private(dev,sizeof(ni_660x_private))) < 0) return ret;

  ret = ni_660x_find_device(dev, it->options[0], it->options[1]);
  if (ret<0) return ret;

  ret = mite_setup(devpriv->mite);
  if (ret < 0) {
    printk("error setting up mite\n");
    return ret;
  }
  dev->iobase = mite_iobase(devpriv->mite);
  dev->board_name = thisboard->name;
  dev->irq = mite_irq(devpriv->mite);
  printk(" %s ", dev->board_name);

  /* Currently there is only 1 subdevice for the encoders,
     and another subdevice for DIO */
  dev->n_subdevices = 2;
	
  if (alloc_subdevices(dev)<0) return -ENOMEM;

  s=dev->subdevices+0;
  /* DIGITAL I/O SUBDEVICE */
  s->type	  = COMEDI_SUBD_DIO;
  s->subdev_flags = SDF_READABLE|SDF_WRITABLE;
  s->n_chan       = 32;
  s->maxdata      = 1;
  s->range_table  = &range_digital;
  /* (Copied from skel.c) DIO devices are slightly special.  Although
   * it is possible to implement the insn_read/insn_write interface,
   * it is much more useful to applications if you implement the
   * insn_bits interface.  This allows packed reading/writing of the
   * DIO channels.  The comedi core can convert between insn_bits and
   * insn_read/write */
  s->insn_bits    = ni_660x_dio_insn_bits;
  s->insn_config  = ni_660x_dio_insn_config;
  s->io_bits      = 0;     /* all bits input */
  /* For Commands?? Not implemented yet...
  s->do_cmd = ni_660x_dio_cmd;
  s->do_cmdtest = ni_660x_dio_cmdtest;
  s->cancel = ni_660x_dio_cancel;
  s->state = 0; KG what does state actually mean?
  */

  s=dev->subdevices+1;
  /* GENERAL-PURPOSE COUNTER/TIME (GPCT) ASIC ENCODER */
  s->type         = COMEDI_SUBD_COUNTER;
  s->subdev_flags = SDF_READABLE | SDF_WRITABLE | SDF_LSAMPL; 
  /* KG: took this from multiq3.c and added SDF_WRITABLE (to latch a
     value in the load register: 
     What does SDF_LSAMPL mean? */
  s->n_chan       = thisboard->n_ctrs;
  s->maxdata      = 0xffffffff; /* 32 bit counter */
  s->insn_read    = ni_660x_GPCT_rinsn;
  s->insn_config  = ni_660x_GPCT_insn_config; 
  s->insn_write   = ni_660x_GPCT_winsn; 
  /* Command are not implemented yet
  s->do_cmd = ni_660x_gpct_cmd;
  s->do_cmdtest = ni_660x_gpct_cmdtest;
  s->cancel = ni_660x_gpct_cancel;
  */

  /* IRQ (not implemented yet)
	 
  Request irq handler:
  
  ret=comedi_request_irq(dev->irq,ni_660x_interrupt,SA_SHIRQ,"ni_660x",dev);      
  if(ret<0)
  {
  dev->irq=0;
  printk(" irq not available");
  }
  */

  printk("attached\n");

  /* What does this "return value" mean?  Is this fixed by API??
     - skel_attach in skel.c returns 1;
     - ni_E_init in ni_mio_common.c returns "0" ... */
  return 1;
}


static int
ni_660x_detach(comedi_device *dev)
{
  printk("comedi%d: ni_660x: remove\n",dev->minor);

  if (dev->private && devpriv->mite)
    mite_unsetup(devpriv->mite);

  /* Free irq */
  /*
    if(dev->irq) comedi_free_irq(dev->irq,dev);
  */

  /* Same question as with attach ... */
  return 0;
}

// Help function: Check what chipset the counter channel is on
int GPCT_check_chipset_from_channel(comedi_device *dev, int channel)
{
  int chipset;
  if ( (channel >= 0) && (channel < CTRS_PER_CHIP) )
    {
      chipset = 0;
    }
  else if ( (channel >= CTRS_PER_CHIP) && (channel < thisboard->n_ctrs) )
    {
      chipset = 1;
      // DPRINTK("NI_660x: Moving to chipset 1\n");
    }
  else
    {
      DPRINTK("NI_660x: Channel specification not between limits\n");
      return -EINVAL;
    }
  return chipset;
}

static int
ni_660x_GPCT_rinsn(comedi_device *dev, comedi_subdevice *s,
		   comedi_insn *insn, lsampl_t *data)
{
  int i; // counts the Data
  int channel = CR_CHAN(insn->chanspec);// Unpack chanspec
  
  int RegisterName; // Which register should we read, depends on channel
  int chipset = GPCT_check_chipset_from_channel(dev,channel);

  /* See Chapter 2.2 Reading Counter Value of the NI Register Level
     Programming Manual: "Reading counter values of armed counters".
     We need to take several measurements to be sure what the counter
     value is 
  */
  int tmpdata[2];
  int address;
  
  /* ============================================================ */
  /* 1 subdevice with 8 channels, differentation based on channel */
  DPRINTK("NI_660x: INSN_READ on channel %d\n", channel);
  
  // Check what Application of Counter this channel is configured for
  switch(ni_660x_gpct_config[channel].App)
    {
    case PositionMeasurement:
      switch(channel)
	{
	case 0: case 4:
	  RegisterName = G0SWSaveRegister;
	  break;
	case 1: case 5:
	  RegisterName = G1SWSaveRegister;
	  break;
	case 2: case 6:
	  RegisterName = G2SWSaveRegister;
	  break;
	case 3: case 7:
	  RegisterName = G3SWSaveRegister;
	  break;
	default: // Impossible
	  return -1;
	}
      // Check if (n > 0)
      if ( insn->n <= 0 )
	{
	  DPRINTK("NI_660X: INSN_READ: n should be > 0\n");
	  return -EINVAL;
	}
      // Now proceed with reading data
      address = dev->iobase 
	+ GPCT_OFFSET[chipset] + 
	registerData[RegisterName].offset;
      for ( i=0 ; i < insn->n ; i++ ) 
	{
	  // We read 32 bits, in case of x86 a long.  Is this
	  // architecture-proof???
	  tmpdata[0] = readl(address);
	  tmpdata[1] = readl(address);
	  if (tmpdata[0] != tmpdata[1])
	    {
	      // In case they differ, the 3d measurement is the
	      // correct value
	      data[i] = readl(address);
	    }
	  // Otherwise, they are the same and the correct counter value
	  else data[i] = tmpdata[0];
	}
      return i;

      // The rest is not implemented yet :-)
    default:
      DPRINTK("NI_660x: INSN_READ: Functionality not implemented yet...\n");
      return -EINVAL;
    }// End switch(ni_660x_gpct_config[channel].App)
}

static int
ni_660x_GPCT_insn_config(comedi_device *dev, comedi_subdevice *s, 
			 comedi_insn *insn, lsampl_t *data)
{
  int channel = CR_CHAN(insn->chanspec);// Unpack chanspec
  int chipset = GPCT_check_chipset_from_channel(dev,channel);
  DPRINTK("NI_660x: INSN_CONFIG: Configuring Channel %d\n",channel);
  
  // Check what type of Counter the user requested, data[0] contains
  // the Application type
  switch(insn->data[0])
    {
    case GPCT_QUADRATURE_ENCODER:
      ni_660x_gpct_config[channel].App = PositionMeasurement;
      /* data[1] contains GPCT_X1, GPCT_X2 or GPCT_X4 */
      switch(insn->data[1])
	{
	case GPCT_X1:
	  (ni_660x_gpct_config[channel]).data[0] = CountingModeQuadX1;
	  break;
	case GPCT_X2:
	  (ni_660x_gpct_config[channel]).data[0] = CountingModeQuadX2;
	  break;
      	case GPCT_X4:
	  (ni_660x_gpct_config[channel]).data[0] = CountingModeQuadX4;
	  break;
	default:
	  DPRINTK("NI_660x: INSN_CONFIG: Wrong Counting mode\n");
	  return -EINVAL;
	}
      // Now finally do something :-)
      /* See P. 3.5 of the Register-Level Programming manual.  This
	 bit has to be set, otherwise, you can't use the second chip.
	 Only God knows why...
      */
      if ( chipset == 1)
	{
	  writel(CounterSwap,dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[ClockConfigRegister].offset);
	}
      else 
	{
	  writel(0x0,dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[ClockConfigRegister].offset);
	}
      
      switch(channel)
	{
	case 0: case 4:
	  // Reset the counter
	  writew(G0Reset,dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G01JointResetRegister].offset);
	  // Disarm
	  writew(Disarm,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G0CommandRegister].offset);
	  // Put 0 as initial counter value in the load register
	  writel(0x0,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G0LoadARegister].offset);
	  // Load (latch) this value into the counter
	  writew(Load,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G0CommandRegister].offset);
	  // Set Counting Mode into X4
	  // Take into account Z pulse (index pulse) only when both channel A
	  // and B are high (arbitrary choice)
	  writew(IndexPhaseHighHigh|(ni_660x_gpct_config[channel]).data[0],
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G0CountingModeRegister].offset);
	  // Arm the counter and put it into Hardware UpDown mode (depending
	  // on the UP/DOWN IO pin: 0 = down
	  writew(UpDownHardware|Arm,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G0CommandRegister].offset);
	  break;
	case 1: case 5:
	  writew(G1Reset,dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G01JointResetRegister].offset);
	  writew(Disarm,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G1CommandRegister].offset);
	  writel(0x0,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G1LoadARegister].offset);
	  writew(Load,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G1CommandRegister].offset);
	  writew(IndexPhaseHighHigh|(ni_660x_gpct_config[channel]).data[0],
		 dev->iobase + GPCT_OFFSET[chipset] 
		 + registerData[G1CountingModeRegister].offset);
	  writew(UpDownHardware|Arm,
		 dev->iobase + GPCT_OFFSET[chipset] 
		 + registerData[G1CommandRegister].offset);
	  break;
	case 2: case 6:
	  writew(G0Reset,dev->iobase + GPCT_OFFSET[chipset] + 
		 registerData[G23JointResetRegister].offset);
	  writew(Disarm,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G1CommandRegister].offset);
	  writel(0x0,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G2LoadARegister].offset);
	  writew(Load,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G2CommandRegister].offset);
	  writew(IndexPhaseHighHigh|(ni_660x_gpct_config[channel]).data[0],
		 dev->iobase + GPCT_OFFSET[chipset] 
		 + registerData[G2CountingModeRegister].offset);
	  writew(UpDownHardware|Arm,
		 dev->iobase + GPCT_OFFSET[chipset] 
		 + registerData[G2CommandRegister].offset);
	  break;
	case 3: case 7:
	  writew(G1Reset,dev->iobase + GPCT_OFFSET[chipset] + 
		 registerData[G23JointResetRegister].offset);
	  writew(Disarm,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G3CommandRegister].offset);
	  writel(0x0,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G3LoadARegister].offset);
	  writew(Load,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G3CommandRegister].offset);
	  writew(IndexPhaseHighHigh|(ni_660x_gpct_config[channel]).data[0],
		 dev->iobase + GPCT_OFFSET[chipset] 
		 + registerData[G3CountingModeRegister].offset);
	  writew(UpDownHardware|Arm,
		 dev->iobase + GPCT_OFFSET[chipset] 
		 + registerData[G3CommandRegister].offset);
	  break;
	default: // Impossible
	  return -1;
	}
      break;
    default:
      DPRINTK("NI_660x: INSN_CONFIG :Your counter ");
      DPRINTK("application is not implemented yet\n");
      return -EINVAL;
    } // End switch(ni_660x_gpct_config[channel])
  return 0;
}

static int ni_660x_GPCT_winsn(comedi_device *dev,
			      comedi_subdevice *s,
			      comedi_insn *insn,
			      lsampl_t * data)
{
  int channel = CR_CHAN(insn->chanspec);// Unpack chanspec
  int chipset = GPCT_check_chipset_from_channel(dev,channel);
  
  DPRINTK("NI_660X: INSN_WRITE on channel %d\n", channel);
  // Check what Application of Counter this channel is configured for
  switch(ni_660x_gpct_config[channel].App)
    {
    case PositionMeasurement:
      switch(channel)
	{
	case 0: case 4:
	  // Disarm the counter
	  writew(Disarm,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G0CommandRegister].offset);
	  // Write the value into the load register
	  writel(*data,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G0LoadARegister].offset);
	  // Latch the value into the counter
	  writew(Load,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G0CommandRegister].offset);
	  // Arm the counter again
	  writew(Arm,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G0CommandRegister].offset);
	  break;
	case 1: case 5:
	  writew(Disarm,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G1CommandRegister].offset);
	  writel(*data,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G1LoadARegister].offset);
	  writew(Load,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G1CommandRegister].offset);
	  writew(Arm,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G1CommandRegister].offset);
	  break;
	case 2: case 6:
	  writew(Disarm,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G2CommandRegister].offset);
	  writel(*data,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G2LoadARegister].offset);
	  writew(Load,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G2CommandRegister].offset);
	  writew(Arm,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G2CommandRegister].offset);
	  break;
	case 3: case 7:
	  writew(Disarm,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G3CommandRegister].offset);
	  writel(*data,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G3LoadARegister].offset);
	  writew(Load,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G3CommandRegister].offset);
	  writew(Arm,
		 dev->iobase 
		 + GPCT_OFFSET[chipset] 
		 + registerData[G3CommandRegister].offset);
	  break;
	default:// Impossible
	  DPRINTK("NI_660X: INSN_WRITE: Invalid channel\n");
	  return -EINVAL;
	}
      break;
    default: // Impossible
      DPRINTK("NI_660X: INSN_WRITE: Functionality %d not implemented yet\n",
	      ni_660x_gpct_config[channel].App);
      return -EINVAL;
    }
  return 0;
}


static int
ni_660x_find_device(comedi_device *dev, int bus, int slot)
{
  struct mite_struct *mite;
  int i;

  for (mite=mite_devices; mite; mite=mite->next) {
    if (mite->used) continue;
    if (bus || slot) {
      if (bus!=mite->pcidev->bus->number ||
	  slot!=PCI_SLOT(mite->pcidev->devfn)) continue;
    }

    for (i=0; i<n_ni_660x_boards; i++) {
      if (mite_device_id(mite)==ni_660x_boards[i].dev_id) {
	dev->board_ptr=ni_660x_boards+i;
	devpriv->mite=mite;
	return 0;
      }
    }
  }
  printk("no device found\n");
  mite_list_devices();
  return -EIO;
}

static int ni_660x_dio_insn_bits(comedi_device *dev,
				 comedi_subdevice *s,
				 comedi_insn *insn,
				 lsampl_t *data)
{
  DPRINTK("NI_660X: INSN_BITS: DIO not implemented yet\n");
  return -EINVAL;
}

static int ni_660x_dio_insn_config(comedi_device *dev,
				   comedi_subdevice *s,
				   comedi_insn *insn,
				   lsampl_t *data)
{
  DPRINTK("NI_660X: INSN_CONFIG: DIO not implemented yet\n");
  return -EINVAL;
}


