/*
 * pci_quad04.c: a driver for the Measurment Computing 
 * 			PCI-QUAD04 Quadurature Encoder interface
 *
 * Copyright (C) 2003, John Conner (conner@empiredi.com)
 *
 * based on procfs_example.c 
 * Copyright (C) 2001, Erik Mouw (J.A.K.Mouw@its.tudelft.nl)
 * and bits from the Linux Device Drivers book.
 *
 *
 * This program is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * This is a very simple driver for the Measurement Computing
 * PCI-QUAD04 four channel quadrature encoder intrerface board.
 * The driver handles the following functions for user space programs:
 * 1. Initialize the board for use.
 * 2. Sets the four encoders for counting.
 * 3. Allows the user to read the current count of each of the four channels.
 * 4. Allows the user to reset the count on each channel.
 * 
 * The driver does not enable or  handle interrupts from the board.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/sched.h>
#include <asm/uaccess.h>
#include <linux/pci.h>
#include <linux/ctype.h>
#include <linux/string.h>


#define MODULE_VERSION "1.0"
#define MODULE_NAME    "pci_quad04"
#define MODULE_DIR     "quad04"

#ifndef PCI_VENDOR_ID_CBOARDS
#define PCI_VENDOR_ID_CBOARDS 0x1307
#endif

#ifndef PCI_DEVICE_ID_CBOARDS_QUAD04
#define PCI_DEVICE_ID_CBOARDS_QUAD04 0x004d
#endif

#define QUAD_LEN 8
#define MAX_IN_LEN 80

static int __devinit quad_probe( struct pci_dev *dev, const struct pci_device_id *quad_dev_id);
static void __devexit quad_remove( struct pci_dev *dev );

/*
 * Defines for registers and bits for the LS7266R1 
 * Quadrature Counter Chip used on the Measurement Computing
 * PCI-QUAD04 board.
 */

/* Primary Registers */
#define X_1DATA(arg)  (arg+0)	// X_1 Data register
#define X_1CMD(arg)   (arg+1)	// X_1 Command register
#define Y_1DATA(arg)  (arg+2)	// Y_1 Data register
#define Y_1CMD(arg)   (arg+3)	// Y_1 Command register
#define X_2DATA(arg)  (arg+4)	// X_2 Data register
#define X_2CMD(arg)   (arg+5)	// X_2 Command register
#define Y_2DATA(arg)  (arg+6)	// Y_2 Data register
#define Y_2CMD(arg)   (arg+7)	// Y_2 Command register

/* Reset and Load Register */
#define  RLD(arg)   (arg | 0x80)// select both counters X&Y
#define XRLD(arg)   (arg | 0 )
#define YRLD(arg)   XRLD(arg)
#define Rst_BP      0x01
#define Rst_CNTR    0x02
#define Rst_FLAGS   0x04
#define Rst_E       0x06
#define Trf_PR_CNTR 0x08
#define Trf_Cntr_OL 0x10
#define Trf_PS0_PSC 0x18

/* Counter Mode Register */
#define  CMR(arg)   (arg | 0xA0)// select both counters X&Y
#define XCMR(arg)   (arg | 0x20)
#define YCMR(arg)   XCMR(arg)
#define BINCnt      0x00	// Count Binary
#define BCDCnt      0x01	// Count BCD
#define NrmCnt      0x00	// Normal Counting
#define RngLmt      0x02	// Range Limit 
#define NRcyc       0x04	// Non-Recycle count
#define ModN        0x06	// Modulo N counting
#define NQDX        0x00	// Non-quad (UP/DN)
#define QDX1        0x08	// Quad X1
#define QDX2        0x10	// Quad X2
#define QDX4        0x18	// Quad X4

/* Input/Output Control Register */
#define  IOR(arg)   (arg | 0xC0)// select both counters X&Y
#define XIOR(arg)   (arg | 0x40)
#define YIOR(arg)   XIOR(arg)
#define DisAB       0x00	// Disable A&B inputs
#define EnAB        0x01	// Enable A&B inputs
#define LCNTR       0x00	// LCNTR/LOL is CNTR input
#define LOL         0x02        // LCNTR/LOL is Load OL input
#define RCNTR       0x00	// RCNTR/ABG is CNTR input
#define ABGate      0x04	// RCNTR/ABG is A&B Enable Gate
#define CYBW        0x00	// FLG1-Carry FLG2-Borrow
#define CPBW        0x08	// FLG1-Compare FLG2-Borrow
#define CB_UPDN     0x10	// FLG1-Carry/Borrow FLG2-UP/DN
#define IDX_ERR     0x18	// FLG1-IDX FLG2-E

/* Index Control Register */
#define  IDR(arg)   (arg | 0xE0)// select both counters X&Y
#define XIDR(arg)   (arg | 0x60)
#define YIDR(arg)   XIDR(arg)
#define DisIDX      0x00	// Disable Index
#define EnIDX       0x01	// Enable Index 
#define NIDX        0x00	// Negative Index Polarity
#define PIDX        0x02	// Positive Index Polarity
#define LIDX        0x00	// LCNTR/LOL pin is Index
#define RIDX        0x04	// RCNTR/ABG pin is Index

/* Interrupt Routing Control Register */
#define IRCR(arg)   (arg + 8)   // Register Address
#define CBINT1      0x10	// Carry/Borrow Interrupt Ch 1
#define CBINT2      0x20	// Carry/Borrow Interrupt Ch 2
#define CBINT3      0x40	// Carry/Borrow Interrupt Ch 3
#define CBINT4      0x80	// Carry/Borrow Interrupt Ch 4
#define IND1SEL     0x01        // Index 1 Select
#define IND2SEL     0x02        // Index 2 Select
#define IND3SEL     0x04        // Index 3 Select
#define IND4SEL     0x08        // Index 4 Select
#define INDSEL      0x0F        // Index All Select

/* Input Signal Control Register */
#define ISCR(arg)   (arg + 9)   // Register Address
#define PH2A        0x01        // FLG1 to PH2A
#define PH2B        0x02        // PH1B to PH2B
#define PH3A        0x04        // FLG2 to PH3A
#define PH3B        0x08        // PH2B to PH3B
#define PH4A        0x10        // FLG3 to PH4A
#define PH4B0       0x20        // PH2B to PH4B
#define PH4B1       0x40        // PH3B to PH4B
#define CNTR_4_24   0x00        // Four 24 bit counters (1/2/3/4)
#define CNTR_1_48   (PH2A+PH2B) // One 48 bit counter Two 24 bit (1-2/3/4)
#define CNTR_2_48   (PH2A+PH2B+PH4A+PH4B1) // Two 48 bit counters (1-2/3-4)
#define CNTR_24_72  (PH3A+PH3B+PH4A+PH4B0) // One 24 (1) and one 72 bit (2-3-4)
#define CNTR_96     (PH2A+PH2B+PH3A+PH3B+PH4A+PH4B0) // One 96 bit (1-2-3-4)

/* Programmable Interrupt Controllers */
#define PIC_A(arg)  (arg + 10)  // PIC Port A
#define PIC_B(arg)  (arg + 11)  // PIC Port B

struct pciquad {
  struct pci_dev *dev;
  unsigned long intrp_base;
  unsigned long intrp_size;
  unsigned long io_base;
  unsigned long io_size;
};

struct quad_data_t {
  struct pciquad *pciquad; // pointer to PCI structure for board
  unsigned int data_port;  // Data port address
  unsigned int cmd_port;   // Command port address
  unsigned int cntr;       // counter index (0 base)
  char name[QUAD_LEN + 1];
  char value[QUAD_LEN + 1];
};

/*
 * Channel register settings
 */
struct chan_regs {
  unsigned long pr;
  unsigned char cmr;
  unsigned char ior;
  unsigned char idr;
};

/*
 * Globla register settings
 */
struct global_regs {
  unsigned char ircr;
  unsigned char iscr;
  unsigned char pica;
  unsigned char picb;
};

static struct proc_dir_entry *quad_dir,
        *cntr_1_file, *cntr_2_file, *cntr_3_file, *cntr_4_file,
	*set_1_file, *set_2_file, *set_3_file, *set_4_file;


static struct quad_data_t cntr_1, cntr_2, cntr_3, cntr_4,
                          set_1, set_2, set_3, set_4;

static struct chan_regs cntr_reg[4];

static struct global_regs global_reg;


/*
 * proc_read_cntr()
 * Handle read requests for the /proc/pci_quad04/cntr_# files.
 * For the files cntr_1, cntr_2, cntr_3 and cntr_4
 * the counter contents are latched and the 24-bit
 * contents of the latch are returned as a decimal 
 * number.
 */
static int proc_read_cntr(char *page, char **start,
    off_t off, int count, int *eof, void *data)
{
  int len;
  unsigned char b_1, b_2, b_3, flags;
  unsigned int  temp;
  struct quad_data_t *quad_data = (struct quad_data_t *)data;
  int data_port = quad_data->data_port;
  int cmd_port  = quad_data->cmd_port;
  int cntr_i    = quad_data->cntr;

  MOD_INC_USE_COUNT;
  
  /* Capture the counter in the output latch and clear the byte pointer.*/
  outb_p( XRLD( Trf_Cntr_OL+Rst_BP ), cmd_port );

  /* Now read the data in a byte at a time */
  b_1   = inb_p( data_port );
  b_2   = inb_p( data_port );
  b_3   = inb_p( data_port );
  flags = inb_p( cmd_port );

  /* combine inputs to form the full count */
  temp = b_3;
  temp <<= 8;
  temp += b_2;
  temp <<= 8;
  temp += b_1;

#ifdef DEBUG /* for debuging */
  len = sprintf(page,
      "%s: %s = %d %#.2x  (%#.2x) ( %#.2x %#.2x %#.2x ) [%#.4x %#.4x] %#.2x %#.2x %#.2x\n",
      quad_data->name, quad_data->value, temp, flags,
      temp, b_3, b_2, b_1,
      data_port, cmd_port, cntr_reg[cntr_i].cmr, cntr_reg[cntr_i].ior,
      cntr_reg[cntr_i].idr );
#else
  len = sprintf(page, "%s = %d %s\n",
      quad_data->value, temp,
      ((flags & 0x20)== 0x20) ? "UP" : "DN" );
#endif

  MOD_DEC_USE_COUNT;

  *eof = 1;
  return len;
}

/*
 * proc_write_cntr()
 * Handle simple writes to the /proc/pci_quad04 files.
 */
static int proc_write_cntr(struct file *file,
    const char *buffer, unsigned long count, void *data)
{
  int len;
  struct quad_data_t *quad_data = (struct quad_data_t *)data;
  struct pciquad *pciquad = quad_data->pciquad;

  MOD_INC_USE_COUNT;

  if(count > QUAD_LEN)
    len = QUAD_LEN;
  else
    len = count;

  if(copy_from_user(quad_data->value, buffer, len))
  {
    MOD_DEC_USE_COUNT;
    return -EFAULT;
  }

  quad_data->value[len] = '\0';

  MOD_DEC_USE_COUNT;

  return len;
}

/*
 * proc_read_set()
 * Handle read requests for the /proc/pci_quad04 files.
 * For the files set_1, set_2, set_3 and set_4
 * the current value of the preset register is returned.
 */
static int proc_read_set(char *page, char **start,
    off_t off, int count, int *eof, void *data)
{
  int len;
  int data_port, cmd_port;
  struct quad_data_t *quad_data = (struct quad_data_t *)data;
  int cntr_i    = quad_data->cntr;

  MOD_INC_USE_COUNT;
  
    len = sprintf(page,
	"%s: %s PR=%d  CMR=%#.2x IOR=%#.2x IDR=%#.2x, ICR=%#.2x, ISR=%#.2x, PIC_A=%#.2x, PIC_B=%#.2x\n", 
		  quad_data->name, quad_data->value,
		  cntr_reg[cntr_i].pr,
		  cntr_reg[cntr_i].cmr, cntr_reg[cntr_i].ior,
		  cntr_reg[cntr_i].idr, global_reg.ircr,
		  global_reg.iscr, global_reg.pica, global_reg.picb);

  MOD_DEC_USE_COUNT;

  return len;
}

/*
 * proc_write_set()
 * Handle setup writes to the quad04 counter registers.
 * The inputs are space delemited pairs of values seperated
 * by an '='.  A full input buffer would contain a string
 * that looks like 
 * "PR=123456 CMR=0x21 IOR=0x40 IDR=0x60"
 * The acceptable pairs are PR=#, CMR=#, IOR=# and IDR=#.
 * The pairs represent the values to be written to the respective
 * register of the counter.
 * The routine accepts 0 or more of the pairs.  At the end of the 
 * routine, the registers of the counter are updated with the 
 * current values.
 */
#define MAX_PARMS 8

static int proc_write_set(struct file *file,
    const char *buffer, unsigned long count, void *data)
{
  int i;
  int len;
  struct quad_data_t *quad_data = (struct quad_data_t *)data;
  struct pciquad *pciquad = quad_data->pciquad;
  int cntr_i    = quad_data->cntr;
  int data_port = quad_data->data_port;
  int cmd_port  = quad_data->cmd_port;
  char strn[MAX_IN_LEN];  // Temporary working buffers
  char token[MAX_IN_LEN];
  char *cp, *strnp, *tok_in[MAX_PARMS], *tok_value[2];
  unsigned char out_byte;

  MOD_INC_USE_COUNT;

  if(count > MAX_IN_LEN)
    len = MAX_IN_LEN;
  else
    len = count;

  if ( len > 0 )
  {
    /* get input from user space */
    if(copy_from_user(strn, buffer, len))
    {
      MOD_DEC_USE_COUNT;
      return -EFAULT;
    }
    printk(KERN_INFO "pci_quad04:write_set() -input %d - %s\n",
	  count, strn );

    *(strn+len) = '\0';

    printk(KERN_INFO "pci_quad04:write_set() -input %d - %s\n",
	  count, strn );

    /*
     * Clean up the input string - all lower case and spaces.
     */
    for ( cp = strn; *cp != '\0'; cp++) 
    {
      if ( isupper(*cp) )
	*cp = tolower(*cp);
      if ( isspace(*cp) )
	*cp = ' ';
    }

    /*
     * Get the input pairs.
     */
    for ( i=0; i<MAX_PARMS; i++ )
    {
      tok_in[i] = NULL;
    }
    for ( i=0; i<MAX_PARMS; i++ )
    {
      tok_in[i] = NULL;
      if ( i == 0 )
      {
	tok_in[i] = strn;
      }
      else
      {
	if (tok_in[i-1] == NULL) 
	  break;
	if ( ( tok_in[i] = strchr( tok_in[i-1], ' ') ) != NULL )
	{
	  *tok_in[i] = '\0';
	  tok_in[i]++;
	}
      }
      printk(KERN_INFO "pci_quad04:write_set() -token %d %#x - %s\n",
	  i, tok_in[i], tok_in[i]);
    }
      

    /*
     * Decode the pairs and store in static area.
     */
    for ( i=0; i<MAX_PARMS; i++ )
    { 
      if ( tok_in[i] == NULL )
	break; // no more tokens

      tok_value[0] = tok_in[i];
      tok_value[1] = strchr( tok_in[i], '=' );
      if ( tok_value[1] == NULL )
	continue;  // not a valid input 
      *tok_value[1] = '\0';
      tok_value[1]++;

      printk(KERN_INFO "pci_quad04:write_set() -token %d %#x - %s - %#x %#x\n",
	  i, tok_in[i], tok_in[i], tok_value[0], tok_value[1] );
      
      if ( strcmp( tok_value[0], "pr" ) == 0 )
      {
	cntr_reg[cntr_i].pr = simple_strtoul( tok_value[1], NULL, 0 );
      }

      if ( strcmp( tok_value[0], "cmr" ) == 0 )
      {
	cntr_reg[cntr_i].cmr =
	  (unsigned char)simple_strtoul( tok_value[1], NULL, 0 );
      }

      if ( strcmp( tok_value[0], "ior" ) == 0 )
      {
	cntr_reg[cntr_i].ior =
	  (unsigned char)simple_strtoul( tok_value[1], NULL, 0 );
      }

      if ( strcmp( tok_value[0], "idr" ) == 0 )
      {
	cntr_reg[cntr_i].idr =
	  (unsigned char)simple_strtoul( tok_value[1], NULL, 0 );
      }

      if ( strcmp( tok_value[0], "ircr" ) == 0 )
      {
	global_reg.ircr =
	  (unsigned char)simple_strtoul( tok_value[1], NULL, 0 );
      }

      if ( strcmp( tok_value[0], "iscr" ) == 0 )
      {
	global_reg.iscr =
	  (unsigned char)simple_strtoul( tok_value[1], NULL, 0 );
      }

      if ( strcmp( tok_value[0], "pica" ) == 0 )
      {
	global_reg.pica =
	  (unsigned char)simple_strtoul( tok_value[1], NULL, 0 );
      }

      if ( strcmp( tok_value[0], "picb" ) == 0 )
      {
	global_reg.picb =
	  (unsigned char)simple_strtoul( tok_value[1], NULL, 0 );
      }

    }

  }

  /*
   * Write the values of the static registers to the device.
   */


  /* Set Reset/Load Register */
  outb( RLD(Rst_CNTR), cmd_port );
  outb( RLD(Rst_FLAGS), cmd_port );
  outb( RLD(Rst_E), cmd_port );
  outb( RLD(Rst_BP), cmd_port );

  /* Set the global registers */
  outb( global_reg.ircr, IRCR(pciquad->io_base) ); 
  outb( global_reg.iscr, ISCR(pciquad->io_base) ); 
  outb( global_reg.pica, PIC_A(pciquad->io_base) ); 
  outb( global_reg.picb, PIC_B(pciquad->io_base) ); 

  /* Counter Mode Register */
  outb( cntr_reg[cntr_i].cmr, cmd_port );

  /* Input/Output Control Register */
  outb( cntr_reg[cntr_i].ior, cmd_port );

  /* Index Register */
  outb( cntr_reg[cntr_i].idr, cmd_port );

  /* Load Counter Preset */
  out_byte = cntr_reg[cntr_i].pr & 0xff; 
  outb( out_byte, data_port );
  out_byte = (cntr_reg[cntr_i].pr >> 8) & 0xff; 
  outb( out_byte, data_port );
  out_byte = (cntr_reg[cntr_i].pr >> 16) & 0xff; 
  outb( out_byte, data_port );

  /* Transfer the load value to the PR register */
  outb( RLD(Trf_PR_CNTR), cmd_port );

  outb( 0x11, cmd_port );
  printk(KERN_INFO "pci_quad04:write_set() - cntr %#2x %#2x %#2x\n",
      inb_p( data_port ), inb_p( data_port ), inb_p( data_port ) );


  MOD_DEC_USE_COUNT;

  return len;
}

/*
 * quad_probe()
 * 1. Initialize the driver structures.
 * 2. Find and initalize the board.
 * 3. Init the 4 channels.
 *    Set to standard quadrature with X1 multiplier.
 *
 */
static int __devinit quad_probe( struct pci_dev *dev,
    const struct pci_device_id *id )
{
  struct pciquad *pciquad;
  int cntr_i;
  int i, err;

  printk(KERN_INFO "pci_quad04:quad_probe() - entered\n");

  if ( ! (pciquad = kmalloc( sizeof( struct pciquad ), GFP_KERNEL ) ) )
    return -1;
  memset( pciquad, 0, sizeof( struct pciquad ) );
  
  pciquad->dev = dev;
  pci_set_drvdata( dev, pciquad );
  
  pci_enable_device( dev );
  
  /*
   * Setup addresses for the registers.
   */
  /* Region 0 - BADR0 - Memory Mapped Config Registers */
  /* Don't think we need access to any of this */
  
  /* Region 1 - BADR1 - I/O Mapped Config Registers - INTCSR */
  pciquad->intrp_base = pci_resource_start( dev, 1 );
  pciquad->intrp_size = pci_resource_len( dev, 1 );
  printk(KERN_INFO "pci_quad04: intrp_base = %x  intrp_size = %d\n",
      pciquad->intrp_base, pciquad->intrp_size );

  if ( (err = check_region(pciquad->intrp_base, pciquad->intrp_size) ) < 0 )
  {
    printk(KERN_INFO "pci_quad04: Intrp check region error - %d\n", err );
    /* clean up the allocated structure */
    kfree( pciquad );
    return err;
  }
  request_region(pciquad->intrp_base, pciquad->intrp_size, "pci_quad04");

  /* Region 2 - BADR2 - I/O Config Registers - 8 Bit */
  pciquad->io_base = pci_resource_start( dev, 2 );
  pciquad->io_size = pci_resource_len( dev, 2 );
  
  printk(KERN_INFO "pci_quad04: io_base = %x  io_size = %d\n",
      pciquad->io_base, pciquad->io_size );

  if ( (err = check_region(pciquad->io_base, pciquad->io_size) ) < 0 )
  {
    printk(KERN_INFO "pci_quad04: I/O check region error - %d\n", err );
    /* clean up the allocated structure */
    kfree( pciquad );
    return err;
  }
  request_region(pciquad->io_base, pciquad->io_size, "pci_quad04");

  printk(KERN_INFO "pci_quad04:quad_probe() - initialize board\n");
  /* Board Initialization */
  /* disable all interrupts */

  /* Initialize the four counter channels */
  outb( CNTR_96, ISCR(pciquad->io_base) ); // 1 Large Counter
  printk(KERN_INFO "pci_quad04:quad_probe() - ISCR (0x3f) - %#2x\n",
      inb(ISCR(pciquad->io_base)) );
  global_reg.iscr =  CNTR_4_24; // 4 Individual Counters
  outb( global_reg.iscr, ISCR(pciquad->io_base) ); 
  printk(KERN_INFO "pci_quad04:quad_probe() - ISCR (0x00) - %#2x\n",
      inb(ISCR(pciquad->io_base)) );

  outb( 0x5a, IRCR(pciquad->io_base) ); 
  printk(KERN_INFO "pci_quad04:quad_probe() - IRCR (0x5a) - %#2x\n",
      inb(IRCR(pciquad->io_base)) );
  global_reg.ircr =  INDSEL; // Index to LCNTR/LOL on all counters
  outb( global_reg.ircr, IRCR(pciquad->io_base) ); // 4 Individual Counters
  printk(KERN_INFO "pci_quad04:quad_probe() - IRCR (0x00) - %#2x\n",
      inb(IRCR(pciquad->io_base)) );

  global_reg.pica = 0;
  global_reg.picb = 0;
  outb( global_reg.pica, PIC_A(pciquad->io_base) ); // Clear PIC Reg A
  outb( global_reg.picb, PIC_B(pciquad->io_base) ); // Clear PIC Reg B

  for ( cntr_i=0; cntr_i < 4; cntr_i++ )
  {
    int data_port = pciquad->io_base + ( 2 * cntr_i );
    int cmd_port  = pciquad->io_base + ( 2 * cntr_i + 1 );

    printk(KERN_INFO "pci_quad04:quad_probe() - cntr_%d - data= %#4x cmd= %#4x\n",
      cntr_i, data_port, cmd_port );
    /* Set Counter Preset to 0 */
    // TODO: do we need to actually out put this?
    cntr_reg[cntr_i].pr = 0;

    /* Set Reset/Load Register */
    outb( XRLD(Rst_CNTR), cmd_port );
    outb( XRLD(Rst_FLAGS), cmd_port );
    outb( XRLD(Rst_E), cmd_port );
    outb( XRLD(Rst_BP), cmd_port );

    /* Be sure the Prescaler is setup */
    outb( 1, data_port );
    outb( XRLD(Trf_PS0_PSC), cmd_port );
    outb( XRLD(Rst_BP), cmd_port );

    /* Set Counter Mode to Normal, Binary, QUADX1 */
    cntr_reg[cntr_i].cmr = XCMR(BINCnt+NrmCnt+QDX1);
    outb( cntr_reg[cntr_i].cmr, cmd_port );

    /* Enable Counter inputs */
    cntr_reg[cntr_i].ior = XIOR(EnAB+ABGate+LOL);
    outb( cntr_reg[cntr_i].ior, cmd_port );

    /* Disable Index */
    cntr_reg[cntr_i].idr =  XIDR(DisIDX);
    outb( cntr_reg[cntr_i].idr, cmd_port );

    /* Load Counter Preset */
    outb( XRLD(Rst_BP), cmd_port );
    outb( 0x12, data_port );
    outb( 0x34, data_port );
    outb( 0x56, data_port );

    /* Transfer the loaded value to the counter */
    outb_p( XRLD(Trf_PR_CNTR), cmd_port );

    /* Transfer counter to latch */
    outb_p( XRLD(Trf_Cntr_OL), cmd_port );
    outb( XRLD(Rst_BP), cmd_port );
    /* now read back the counter */
    printk(KERN_INFO "pci_quad04:quad_probe() - cntr_%d test %#2x %#2x %#2x\n",
	cntr_i, inb_p( data_port ), inb_p( data_port ), inb_p( data_port ) );
    printk(KERN_INFO "pci_quad04:quad_probe() - cntr_%d status %#2x\n",
	cntr_i, inb_p( cmd_port ) );

    /* Clear the preset */
    outb( XRLD(Rst_BP), cmd_port );
    outb( 0, data_port );
    outb( 0, data_port );
    outb( 0, data_port );

    /* reset every thing */
    outb( XRLD(Rst_CNTR), cmd_port );
    outb( XRLD(Rst_FLAGS), cmd_port );
    outb( XRLD(Rst_E), cmd_port );
    outb( XRLD(Rst_BP), cmd_port );

  }

  /* Initialize the four counter data structures */

  /* Counter 1 */
  cntr_1.pciquad   = pciquad;
  cntr_1.data_port = X_1DATA(pciquad->io_base);
  cntr_1.cmd_port  = X_1CMD(pciquad->io_base);
  cntr_1.cntr      = 0;
  strcpy(cntr_1.name, "cntr_1");
  strcpy(cntr_1.value, "cntr_1");

  /* Counter 2 */
  cntr_2.pciquad   = pciquad;
  cntr_2.data_port = Y_1DATA(pciquad->io_base);
  cntr_2.cmd_port  = Y_1CMD(pciquad->io_base);
  cntr_2.cntr      = 1;
  strcpy(cntr_2.name, "cntr_2");
  strcpy(cntr_2.value, "cntr_2");

  /* Counter 3 */
  cntr_3.pciquad   = pciquad;
  cntr_3.data_port = X_2DATA(pciquad->io_base);
  cntr_3.cmd_port  = X_2CMD(pciquad->io_base);
  cntr_3.cntr      = 2;
  strcpy(cntr_3.name, "cntr_3");
  strcpy(cntr_3.value, "cntr_3");

  /* Counter 4 */
  cntr_4.pciquad   = pciquad;
  cntr_4.data_port = Y_2DATA(pciquad->io_base);
  cntr_4.cmd_port  = Y_2CMD(pciquad->io_base);
  cntr_4.cntr      = 3;
  strcpy(cntr_4.name, "cntr_4");
  strcpy(cntr_4.value, "cntr_4");

  /* Settings 1 */
  set_1.pciquad   = pciquad;
  set_1.data_port = X_1DATA(pciquad->io_base);
  set_1.cmd_port  = X_1CMD(pciquad->io_base);
  set_1.cntr      = 0;
  strcpy(set_1.name, "set_1");
  strcpy(set_1.value, "set_1");

  /* Settings 2 */
  set_2.pciquad   = pciquad;
  set_2.data_port = Y_1DATA(pciquad->io_base);
  set_2.cmd_port  = Y_1CMD(pciquad->io_base);
  set_2.cntr      = 1;
  strcpy(set_2.name, "set_2");
  strcpy(set_2.value, "set_2");

  /* Settings 3 */
  set_3.pciquad   = pciquad;
  set_3.data_port = X_2DATA(pciquad->io_base);
  set_3.cmd_port  = X_2CMD(pciquad->io_base);
  set_3.cntr      = 2;
  strcpy(set_3.name, "set_3");
  strcpy(set_3.value, "set_3");

  /* Settings 4 */
  set_4.pciquad   = pciquad;
  set_4.data_port = Y_2DATA(pciquad->io_base);
  set_4.cmd_port  = Y_2CMD(pciquad->io_base);
  set_4.cntr      = 3;
  strcpy(set_4.name, "set_4");
  strcpy(set_4.value, "set_4");

  printk(KERN_INFO "pci_quad04:quad_probe() - done\n");
  return 0; 
}

static void __devexit quad_remove( struct pci_dev *dev )
{
  struct pciquad *pciquad = pci_get_drvdata( dev );

  release_region(pciquad->intrp_base, pciquad->intrp_size);
  release_region(pciquad->io_base, pciquad->io_size);
  kfree( pciquad );
}

/*
 * Setup the device ID structure to look for a PCI-QUAD04.
 */
struct pci_device_id quad_id_table[ ] __devinitdata =
{
  { PCI_VENDOR_ID_CBOARDS, PCI_DEVICE_ID_CBOARDS_QUAD04,
    PCI_VENDOR_ID_CBOARDS, PCI_DEVICE_ID_CBOARDS_QUAD04,
    0, 0, 0L
  },
  { 0 }
};

/*
 * Setup the pci_driver structure.
 */
struct pci_driver    quad_dvr = {
	name:        "PCI_QUAD04",
	id_table:    quad_id_table,
	probe:       quad_probe,
	remove:      __devexit_p(quad_remove),
};

/*
 * Initialize the driver and board.
 */
static int __init init_pci_quad04(void)
{
  int rv = 0;

  /*
   * Find and initialize the board.
   */
  if ( pci_module_init(&quad_dvr) != 0 )
  {
    printk(KERN_INFO "pci_quad04:init_pci_quad04:pci_module_init() - failed\n");
    rv = -ENODEV;
    goto out;
  }

  /* create directory */
  quad_dir = proc_mkdir(MODULE_NAME, NULL);
  if(quad_dir == NULL)
  {
	  rv = -ENOMEM;
	  goto out;
  }
  /* set owner of directory */
  quad_dir->owner = THIS_MODULE;

  /*
   * create cntr_1, cntr_2, cntr_3 and cntr_4 files
   * using same callback functions 
   */
  cntr_1_file = create_proc_entry("cntr_1", 0666, quad_dir);
  if(cntr_1_file == NULL)
  {
    rv = -ENOMEM;
    goto no_cntr_1;
  }
  cntr_1_file->data = &cntr_1;
  cntr_1_file->read_proc = proc_read_cntr;
  cntr_1_file->write_proc = proc_write_cntr;
  cntr_1_file->owner = THIS_MODULE;
	  
  cntr_2_file = create_proc_entry("cntr_2", 0666, quad_dir);
  if(cntr_2_file == NULL)
  {
    rv = -ENOMEM;
    goto no_cntr_2;
  }
  cntr_2_file->data = &cntr_2;
  cntr_2_file->read_proc = proc_read_cntr;
  cntr_2_file->write_proc = proc_write_cntr;
  cntr_2_file->owner = THIS_MODULE;
	  
  cntr_3_file = create_proc_entry("cntr_3", 0666, quad_dir);
  if(cntr_3_file == NULL)
  {
    rv = -ENOMEM;
    goto no_cntr_3;
  }
  cntr_3_file->data = &cntr_3;
  cntr_3_file->read_proc = proc_read_cntr;
  cntr_3_file->write_proc = proc_write_cntr;
  cntr_3_file->owner = THIS_MODULE;
	  
  cntr_4_file = create_proc_entry("cntr_4", 0666, quad_dir);
  if(cntr_4_file == NULL)
  {
    rv = -ENOMEM;
    goto no_cntr_4;
  }
  cntr_4_file->data = &cntr_4;
  cntr_4_file->read_proc = proc_read_cntr;
  cntr_4_file->write_proc = proc_write_cntr;
  cntr_4_file->owner = THIS_MODULE;
	  
  /*
   * create set_1, set_2, set_3 and set_4 files
   * using same callback functions 
   */
  set_1_file = create_proc_entry("set_1", 0666, quad_dir);
  if(set_1_file == NULL)
  {
    rv = -ENOMEM;
    goto no_set_1;
  }
  set_1_file->data = &set_1;
  set_1_file->read_proc = proc_read_set;
  set_1_file->write_proc = proc_write_set;
  set_1_file->owner = THIS_MODULE;
	  
  set_2_file = create_proc_entry("set_2", 0666, quad_dir);
  if(set_2_file == NULL)
  {
    rv = -ENOMEM;
    goto no_set_2;
  }
  set_2_file->data = &set_2;
  set_2_file->read_proc = proc_read_set;
  set_2_file->write_proc = proc_write_set;
  set_2_file->owner = THIS_MODULE;
	  
  set_3_file = create_proc_entry("set_3", 0666, quad_dir);
  if(set_3_file == NULL)
  {
    rv = -ENOMEM;
    goto no_set_3;
  }
  set_3_file->data = &set_3;
  set_3_file->read_proc = proc_read_set;
  set_3_file->write_proc = proc_write_set;
  set_3_file->owner = THIS_MODULE;
	  
  set_4_file = create_proc_entry("set_4", 0666, quad_dir);
  if(set_4_file == NULL)
  {
    rv = -ENOMEM;
    goto no_set_4;
  }
  set_4_file->data = &set_4;
  set_4_file->read_proc = proc_read_set;
  set_4_file->write_proc = proc_write_set;
  set_4_file->owner = THIS_MODULE;
	  
  /* everything OK */
  printk(KERN_INFO "%s %s initialised\n",
	 MODULE_NAME, MODULE_VERSION);
  return 0;

/*
 * Clean up any thing we've completed on errors.
 */
no_set_4:
	remove_proc_entry("set_3", quad_dir);
no_set_3:
	remove_proc_entry("set_2", quad_dir);
no_set_2:
	remove_proc_entry("set_1", quad_dir);
no_set_1:
	remove_proc_entry("cntr_4", quad_dir);
no_cntr_4:
	remove_proc_entry("cntr_3", quad_dir);
no_cntr_3:
	remove_proc_entry("cntr_2", quad_dir);
no_cntr_2:
	remove_proc_entry("cntr_1", quad_dir);
no_cntr_1:
	remove_proc_entry(MODULE_NAME, NULL);
out:
	return rv;
}


static void __exit cleanup_pci_quad04(void)
{
  remove_proc_entry("set_4", quad_dir);
  remove_proc_entry("set_3", quad_dir);
  remove_proc_entry("set_2", quad_dir);
  remove_proc_entry("set_1", quad_dir);
  remove_proc_entry("cntr_4", quad_dir);
  remove_proc_entry("cntr_3", quad_dir);
  remove_proc_entry("cntr_2", quad_dir);
  remove_proc_entry("cntr_1", quad_dir);
  remove_proc_entry(MODULE_NAME, NULL);

  pci_unregister_driver(&quad_dvr);

  printk(KERN_INFO "%s %s removed\n", MODULE_NAME, MODULE_VERSION);
}


module_init(init_pci_quad04);
module_exit(cleanup_pci_quad04);

MODULE_AUTHOR("John Conner");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Measurment Computing PCI-QUAD04 driver");

EXPORT_NO_SYMBOLS;
